#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-mothershot.h"

void getofc_mothershot(struct vg3_ofunc_objfunc *);

static struct vg3_ofunc_object * f_new(void *, unsigned int, ...);
static void f_free(void *, struct vg3_ofunc_object *);
static void f_run(void *, struct vg3_ofunc_object *);
static void f_draw(void *, struct vg3_ofunc_object *);


/* +++ get-function +++ */

void
getofc_mothershot(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_MOTHERSHOT));
  ofc->f_new = f_new;
  ofc->f_free = f_free;
  ofc->f_run = f_run;
  ofc->f_draw = f_draw;
}


/* +++ object-instance-functions +++ */

/* new-function
 * variable parameters:
 * - int: x-position (centered)
 * - int: y-position (centered)
 * - int: x-moving in 1/100-pixels
 * - int: y-moving in 1/100-pixels
 */
static struct vg3_ofunc_object *
f_new(void *vmain, unsigned int iparent, ...)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_mothershot *gobj;
  struct vg3_coll coll;
  va_list ap;
  int xm, ym, xd, yd;

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* get arguments */
  va_start(ap, iparent);
  xm = va_arg(ap, int);
  ym = va_arg(ap, int);
  xd = va_arg(ap, int);
  yd = va_arg(ap, int);
  va_end(ap);

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }

  gobj->iparent = iparent;

  /* load image */
  gobj->img= VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/mothership/shot.bmp", 1);
  if (gobj->img == NULL) { return NULL; }

  /* set position and moving */
  VG3_image_getsize(gmain->wstruct, gobj->img, NULL, &gobj->rect.w, &gobj->rect.h);
  gobj->rect.x = xm - gobj->rect.w / 2;
  gobj->rect.y = ym - gobj->rect.h / 2;
  gobj->xdelta = xd;
  gobj->ydelta = yd;

  /* set strength */
  gobj->strength = 3;

  /* create and fill mothershot-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_MOTHERSHOT));
  objp->subid = 0;
  objp->drawlevel = 4;
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* insert mothershot-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  /* insert mothershot-instance into quadtree */
  memset(&coll, 0, sizeof(coll));
  coll.rect = gobj->rect;
  snprintf(coll.oid, sizeof(coll.oid), "%s", objp->oid);
  coll.optr = objp;
  VG3_coll_q_insert(gmain->qdtr, &coll);

  return objp;
}


/* free-function */
static void
f_free(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_mothershot *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_mothershot *)objp->ostruct;

  /* inform parent object-instance */
  if (gobj->iparent > 0) {
    struct vg3_ofunc_object *oparent = VG3_ofunc_objlist_isvalid(gmain->ofstruct, gobj->iparent);
    if (oparent != NULL) {
      const struct vg3_ofunc_objobjfunc *oofc;
      oofc = VG3_ofunc_get_objobjfunc(gmain->ofstruct, objp->oid, oparent->oid);
      if (oofc != NULL && oofc->f_quit != NULL) {
        oofc->f_quit(gmain, objp, oparent);
      }
    }
  }

  /* remove mothershot-instance from quadtree */
  VG3_coll_q_remove(gmain->qdtr, objp);

  /* remove mothershot-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  /* free mothershot-instance */
  if (gobj->img != NULL) { VG3_image_unload(gmain->wstruct, gobj->img); }
  free(gobj);
  free(objp);
}


/* run-function */
static void
f_run(void *vmain, struct vg3_ofunc_object *objp)
{
  const int moving_factor = 10;
  struct g_main *gmain = vmain;
  struct g_obj_mothershot *gobj;
  struct vg3_coll coll;
  int erg;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_mothershot *)objp->ostruct;

  /* remove mothershot-instance from quadtree */
  VG3_coll_q_remove(gmain->qdtr, objp);

  /* destroy if outside window or strength = 0 */
  if (gobj->rect.x < -gobj->rect.w) { f_free(gmain, objp); return; }
  if (gobj->rect.y < -gobj->rect.h || gobj->rect.y > gmain->winh) { f_free(gmain, objp); return; }
  if (gobj->strength == 0) { f_free(gmain, objp); return; }

  /* move mothershot-instance and check for collisions with gobj->rect */
  erg = VG3_move_object_check_collision(gmain, gmain->ofstruct, gmain->qdtr, objp, &gobj->rect, moving_factor, &gobj->xdelta, &gobj->ydelta, &gobj->xremainder, &gobj->yremainder);
  if (erg < 0) {  /* error: destroy mothershot-instance */
    fprintf(stderr, "moving mothershot: %s\n", VG3_error());
    f_free(gmain, objp);
    return;
  }
  if (erg == VGAG3_COLL_RETURN_DEAD) { return; }  /* instance has been freed */
  if (erg == VGAG3_COLL_RETURN_CATCHED) { f_free(gmain, objp); return; }

  /* insert mothershot-instance into quadtree again */
  memset(&coll, 0, sizeof(coll));
  coll.rect = gobj->rect;
  snprintf(coll.oid, sizeof(coll.oid), "%s", objp->oid);
  coll.optr = objp;
  VG3_coll_q_insert(gmain->qdtr, &coll);
}


/* draw-function */
static void
f_draw(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_mothershot *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_mothershot *)objp->ostruct;

  /* draw mothershot-instance */
  VG3_image_copy(gmain->wstruct, NULL, gobj->img, gobj->rect.x + gobj->rect.w / 2, gobj->rect.y + gobj->rect.h / 2, NULL, 0);
}
