#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-player.h"

#define MAX_HEALTH 5

void getofc_player(struct vg3_ofunc_objfunc *);

static struct vg3_ofunc_object * f_new(void *, unsigned int, ...);
static void f_free(void *, struct vg3_ofunc_object *);
static void f_run(void *, struct vg3_ofunc_object *);
static void f_draw(void *, struct vg3_ofunc_object *);
static int f_data(void *, struct vg3_ofunc_object *, void *);


/* +++ get-function +++ */

void
getofc_player(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_PLAYER));
  ofc->f_new = f_new;
  ofc->f_free = f_free;
  ofc->f_run = f_run;
  ofc->f_draw = f_draw;
  ofc->f_data = f_data;
}


/* +++ object-instance-functions +++ */

/* new-function
 * variable parameters: none
 */
static struct vg3_ofunc_object *
f_new(void *vmain, unsigned int iparent, ...)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_player *gobj;
  struct vg3_coll coll;
  va_list ap;

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* get arguments */
  va_start(ap, iparent);
  va_end(ap);

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }

  /* load image and hit-sound */
  gobj->images.img_normal = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/player/player.bmp", 1);
  if (gobj->images.img_normal == NULL) { return NULL; }
  gobj->images.img_up = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/player/player-up.bmp", 1);
  if (gobj->images.img_up == NULL) { return NULL; }
  gobj->images.img_down = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/player/player-down.bmp", 1);
  if (gobj->images.img_down == NULL) { return NULL; }

  gobj->images.exhaust = VG3_sprite_load(gmain->wstruct, FILES_DIR "/bmp/player/exhaust.sprite");
  if (gobj->images.exhaust == NULL) { return NULL; }
  gobj->images.hit = VG3_sprite_load(gmain->wstruct, FILES_DIR "/bmp/hit/hit.sprite");
  if (gobj->images.hit == NULL) { return NULL; }
  gobj->images.expl = VG3_sprite_load(gmain->wstruct, FILES_DIR "/bmp/player/expl.sprite");
  if (gobj->images.expl == NULL) { return NULL; }

  gobj->hitkz = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/hit.wav", 100, VGAG3_AUDIO_VOLUME_SOUND);

  /* set position */
  gobj->draw.img = gobj->images.img_normal;
  VG3_image_getsize(gmain->wstruct, gobj->draw.img, NULL, &gobj->rect.w, &gobj->rect.h);
  gobj->rect.x = gobj->rect.w;
  gobj->rect.y = (gmain->winh - gobj->rect.h) / 2;
  /* set collision rectangle with 80% of width and height of image-rectangle */
  gobj->crect = gobj->rect;
  gobj->crect = VG3_correct_imageposition(&gobj->crect, gobj->rect.w * 80 / 100, gobj->rect.h * 80 / 100);

  /* set health */
  gobj->health = MAX_HEALTH;

  /* create and fill player-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_PLAYER));
  objp->drawlevel = 5;
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* insert player-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  /* insert player-instance (collision-rectangle) into quadtrees */
  memset(&coll, 0, sizeof(coll));
  coll.rect = gobj->crect;
  snprintf(coll.oid, sizeof(coll.oid), "%s", objp->oid);
  coll.optr = objp;
  VG3_coll_q_insert(gmain->qdtr, &coll);
  VG3_coll_q_insert(gmain->game.qdtr_long, &coll);

  return objp;
}


/* free-function */
static void
f_free(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_player *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_player *)objp->ostruct;

  /* remove player-instance from quadtrees */
  VG3_coll_q_remove(gmain->qdtr, objp);
  VG3_coll_q_remove(gmain->game.qdtr_long, objp);

  /* remove player-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  /* free player-instance */
  if (gobj->images.img_normal != NULL) { VG3_image_unload(gmain->wstruct, gobj->images.img_normal); }
  if (gobj->images.img_up != NULL) { VG3_image_unload(gmain->wstruct, gobj->images.img_up); }
  if (gobj->images.img_down != NULL) { VG3_image_unload(gmain->wstruct, gobj->images.img_down); }
  if (gobj->images.exhaust != NULL) { VG3_sprite_free(gobj->images.exhaust); }
  if (gobj->images.hit != NULL) { VG3_sprite_free(gobj->images.hit); }
  if (gobj->images.expl != NULL) { VG3_sprite_free(gobj->images.expl); }
  if (gobj->hitkz > 0) { VG3_audio_unload(gmain->wstruct, gobj->hitkz); }
  free(gobj);
  free(objp);
}


/* run-function */
static void
f_run(void *vmain, struct vg3_ofunc_object *objp)
{
  const int moving_factor = 50;
  const int moving_percent = 100;
  struct g_main *gmain = vmain;
  struct g_obj_player *gobj;
  struct vg3_coll coll;
  int shoot, erg;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_player *)objp->ostruct;

  /* remove player-instance from quadtrees */
  VG3_coll_q_remove(gmain->qdtr, objp);
  VG3_coll_q_remove(gmain->game.qdtr_long, objp);

  /* check if player is exploding */
  if (gobj->health <= 0) { return; }

  /* check for key-pressings */
  shoot = 0;
  gobj->xdelta = gobj->ydelta = 0;
  if (VG3_keys_key_ispressed(gmain->skeys, NULL, 0, -1, KEYDEF_GO_FORWARD, VGAG3_IS_PRESSED)) {
    if (gobj->rect.x < gmain->winw - gobj->rect.w) { gobj->xdelta = moving_percent; }
  }
  if (VG3_keys_key_ispressed(gmain->skeys, NULL, 0, -1, KEYDEF_GO_BACKWARD, VGAG3_IS_PRESSED)) {
    if (gobj->rect.x > 0) { gobj->xdelta = -moving_percent; }
  }
  if (VG3_keys_key_ispressed(gmain->skeys, NULL, 0, -1, KEYDEF_GO_UP, VGAG3_IS_PRESSED)) {
    if (gobj->rect.y > 0) { gobj->ydelta = -moving_percent; }
  }
  if (VG3_keys_key_ispressed(gmain->skeys, NULL, 0, -1, KEYDEF_GO_DOWN, VGAG3_IS_PRESSED)) {
    if (gobj->rect.y < gmain->winh - gobj->rect.h) { gobj->ydelta = moving_percent; }
  }
  if (VG3_keys_key_ispressed(gmain->skeys, NULL, 0, -1, KEYDEF_FIRE_LASER, VGAG3_IS_NEW_PRESSED)) {
    if (gobj->shot_running < 3) { shoot = PLAYERSHOT_SHOT; }
  }
  if (VG3_keys_key_ispressed(gmain->skeys, NULL, 0, -1, KEYDEF_FIRE_TORPEDO, VGAG3_IS_NEW_PRESSED)) {
    if (gobj->torpedo_strength > 0 && gobj->torpedo_running < 1) { shoot = PLAYERSHOT_TORPEDO; }
  }

  /* pointer to actual image */
  if (gobj->ydelta < 0) {
    gobj->draw.img = gobj->images.img_up;
  } else if (gobj->ydelta > 0) {
    gobj->draw.img = gobj->images.img_down;
  } else {
    gobj->draw.img = gobj->images.img_normal;
  }

  /* check for drawing exhaust */
  if (gobj->xdelta > 0) {
    gobj->draw.exhaust = 1;
  } else if (gobj->draw.exhaust) {
    VG3_sprite_rewind(gobj->images.exhaust);
    gobj->draw.exhaust = 0;
  }

  /* move player-instance and check for collisions with gobj->crect */
  erg = VG3_move_object_check_collision(gmain, gmain->ofstruct, gmain->qdtr, objp, &gobj->crect, moving_factor, &gobj->xdelta, &gobj->ydelta, &gobj->xremainder, &gobj->yremainder);
  if (erg < 0) {  /* error: destroy player-instance */
    fprintf(stderr, "moving player: %s\n", VG3_error());
    f_free(gmain, objp);
    return;
  }
  if (erg == VGAG3_COLL_RETURN_DEAD) { return; }  /* instance has been freed */
  /* VGAG3_COLL_RETURN_CATCHED: ignore (and hope to come free) */

  /* x/y-value of gobj->crect could be modified, correct it also in gobj->rect */
  gobj->rect = VG3_correct_imageposition(&gobj->crect, gobj->rect.w, gobj->rect.h);

  /* release shot */
  if (shoot) {
    const struct vg3_ofunc_objfunc *ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_PLAYERSHOT));
    if (ofc != NULL) {
      int speed, strength;
      if (shoot == PLAYERSHOT_TORPEDO) {
        speed = moving_factor * 2 / 10;
        strength = gobj->torpedo_strength;
      } else {
        speed = moving_factor * 3 / 10;
        strength = 1;
      }
      if (ofc->f_new(gmain, objp->instanceid, gobj->rect.x + gobj->rect.w, gobj->rect.y + gobj->rect.h / 2, speed, 0, shoot, strength) != NULL) {
        if (shoot == PLAYERSHOT_TORPEDO) { gobj->torpedo_running++; } else { gobj->shot_running++; }
      }
    }
  }

  /* check long-distance-quadtree */
  if (check_long_distance(gmain, objp, &gobj->crect) < 0) {
    fprintf(stderr, "player: check long-distance-quadtree: %s\n", VG3_error());
    f_free(gmain, objp);
    return;
  }

  /* insert player-instance (collision-rectangle) into quadtrees again */
  memset(&coll, 0, sizeof(coll));
  coll.rect = gobj->crect;
  snprintf(coll.oid, sizeof(coll.oid), "%s", objp->oid);
  coll.optr = objp;
  VG3_coll_q_insert(gmain->qdtr, &coll);
  VG3_coll_q_insert(gmain->game.qdtr_long, &coll);
}


/* draw-function */
static void
f_draw(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_player *gobj;
  struct vg3_image *img;
  struct vg3_image_attributes iattr;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_player *)objp->ostruct;

  /* draw player-instance */
  if (gobj->health <= 0) {  /* exploding */
    if (VG3_sprite_get(gobj->images.expl, &img, &iattr, NULL)) {
      if (img != NULL) {
        VG3_image_copy(gmain->wstruct, NULL, img, gobj->rect.x + gobj->rect.w / 2, gobj->rect.y + gobj->rect.h / 2, &iattr, 0);
      }
    } else {  /* free player */
      f_free(gmain, objp);
      gmain->game.level_retval = -1;  /* game failed */
      return;
    }
  } else {
    VG3_image_copy(gmain->wstruct, NULL, gobj->draw.img, gobj->rect.x + gobj->rect.w / 2, gobj->rect.y + gobj->rect.h / 2, NULL, 0);
    if (gobj->draw.exhaust) {
      if (VG3_sprite_get(gobj->images.exhaust, &img, &iattr, NULL) && img != NULL) {
        int w1;
        VG3_image_getsize(gmain->wstruct, img, NULL, &w1, NULL);
        VG3_image_copy(gmain->wstruct, NULL, img, gobj->rect.x - 2 - w1 / 2, gobj->rect.y + gobj->rect.h / 2, &iattr, 0);
      }
    }
    if (gobj->draw.hit) {  /* being hit */
      if (!VG3_sprite_get(gobj->images.hit, &img, &iattr, NULL)) {
        VG3_sprite_rewind(gobj->images.hit);
        gobj->draw.hit = 0;
      } else if (img != NULL) {
        VG3_image_copy(gmain->wstruct, NULL, img, gobj->rect.x + gobj->rect.w / 2, gobj->rect.y + gobj->rect.h / 2, &iattr, 0);
      }
    }
  }
  draw_health(vmain, MAX_HEALTH, gobj->health, 1);
}


/* data-exchanging-function */
static int
f_data(void *vmain, struct vg3_ofunc_object *objp, void *vdata)
{
  struct g_main *gmain = vmain;
  struct fdata_number *fdnumber = (struct fdata_number *)vdata;
  struct g_obj_player *gobj;

  if (gmain == NULL || objp == NULL || fdnumber == NULL) { return 0; }

  gobj = (struct g_obj_player *)objp->ostruct;

  if (fdnumber->flag == FDATA_REINIT) {  /* reinitialize after level-increment */
    /* set position */
    gobj->draw.img = gobj->images.img_normal;
    gobj->rect.x = gobj->rect.w;
    gobj->rect.y = (gmain->winh - gobj->rect.h) / 2;
    /* set collision rectangle with 80% of width and height of image-rectangle */
    gobj->crect = gobj->rect;
    gobj->crect = VG3_correct_imageposition(&gobj->crect, gobj->rect.w * 80 / 100, gobj->rect.h * 80 / 100);
    /* set health */
    gobj->health = MAX_HEALTH;
    /* rewind sprites */
    VG3_sprite_rewind(gobj->images.hit);
    VG3_sprite_rewind(gobj->images.exhaust);
  } else if (fdnumber->flag == FDATA_TORPEDO) {  /* set torpedo_strength */
    gobj->torpedo_strength = fdnumber->number1;
  } else if (fdnumber->flag == FDATA_POS) {  /* return center-position */
    fdnumber->number1 = gobj->rect.x + gobj->rect.w / 2;
    fdnumber->number2 = gobj->rect.y + gobj->rect.h / 2;
  }

  return 1;
}
