#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-playershot.h"

void getofc_playershot(struct vg3_ofunc_objfunc *);

static struct vg3_ofunc_object * f_new(void *, unsigned int, ...);
static void f_free(void *, struct vg3_ofunc_object *);
static void f_run(void *, struct vg3_ofunc_object *);
static void f_draw(void *, struct vg3_ofunc_object *);


/* +++ get-function +++ */

void
getofc_playershot(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_PLAYERSHOT));
  ofc->f_new = f_new;
  ofc->f_free = f_free;
  ofc->f_run = f_run;
  ofc->f_draw = f_draw;
}


/* +++ object-instance-functions +++ */

/* new-function
 * variable parameters:
 * - int: x-position (centered)
 * - int: y-position (centered)
 * - int: x-moving in pixels
 * - int: y-moving in pixels
 * - int: one from PLAYERSHOT_*
 * - int: strength
 */
static struct vg3_ofunc_object *
f_new(void *vmain, unsigned int iparent, ...)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_playershot *gobj;
  struct vg3_coll coll;
  va_list ap;
  int xm, ym, xd, yd, subid, strength;

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* get arguments */
  va_start(ap, iparent);
  xm = va_arg(ap, int);
  ym = va_arg(ap, int);
  xd = va_arg(ap, int);
  yd = va_arg(ap, int);
  subid = va_arg(ap, int);
  strength = va_arg(ap, int);
  va_end(ap);

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }

  gobj->iparent = iparent;

  /* load image and hit-sound */
  if (subid == PLAYERSHOT_TORPEDO) {
    gobj->img= VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/player/torpedo.bmp", 1);
  } else {
    gobj->img= VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/player/shot.bmp", 1);
  }
  if (gobj->img == NULL) { return NULL; }
  gobj->hitkz = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/hit.wav", 100, VGAG3_AUDIO_VOLUME_SOUND);

  /* set position and moving */
  VG3_image_getsize(gmain->wstruct, gobj->img, NULL, &gobj->rect.w, &gobj->rect.h);
  gobj->rect.x = xm - gobj->rect.w / 2;
  gobj->rect.y = ym - gobj->rect.h / 2;
  gobj->xdelta = xd * 100;
  gobj->ydelta = yd * 100;

  /* set strength */
  if (strength < 1) { strength = 1; }
  gobj->strength = strength;

  /* create and fill playershot-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_PLAYERSHOT));
  objp->subid = subid;
  objp->drawlevel = 4;
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* insert playershot-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  /* insert playershot-instance into quadtree */
  memset(&coll, 0, sizeof(coll));
  coll.rect = gobj->rect;
  snprintf(coll.oid, sizeof(coll.oid), "%s", objp->oid);
  coll.optr = objp;
  VG3_coll_q_insert(gmain->qdtr, &coll);

  return objp;
}


/* free-function */
static void
f_free(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_playershot *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_playershot *)objp->ostruct;

  /* inform parent object-instance */
  if (gobj->iparent > 0) {
    struct vg3_ofunc_object *oparent = VG3_ofunc_objlist_isvalid(gmain->ofstruct, gobj->iparent);
    if (oparent != NULL) {
      const struct vg3_ofunc_objobjfunc *oofc;
      oofc = VG3_ofunc_get_objobjfunc(gmain->ofstruct, objp->oid, oparent->oid);
      if (oofc != NULL && oofc->f_quit != NULL) {
        oofc->f_quit(gmain, objp, oparent);
      }
    }
  }

  /* remove playershot-instance from quadtree */
  VG3_coll_q_remove(gmain->qdtr, objp);

  /* remove playershot-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  /* free playershot-instance */
  if (gobj->img != NULL) { VG3_image_unload(gmain->wstruct, gobj->img); }
  if (gobj->hitkz > 0) { VG3_audio_unload(gmain->wstruct, gobj->hitkz); }
  free(gobj);
  free(objp);
}


/* run-function */
static void
f_run(void *vmain, struct vg3_ofunc_object *objp)
{
  const int moving_factor = 10;
  struct g_main *gmain = vmain;
  struct g_obj_playershot *gobj;
  struct vg3_coll coll;
  int erg;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_playershot *)objp->ostruct;

  /* remove playershot-instance from quadtree */
  VG3_coll_q_remove(gmain->qdtr, objp);

  /* destroy if outside window or strength = 0 */
  if (gobj->rect.x > gmain->winw) { f_free(gmain, objp); return; }
  if (gobj->strength == 0) {
    if (gobj->hitkz == 0 || !VG3_audio_isplaying(gmain->wstruct, gobj->hitkz)) { f_free(gmain, objp); }
    return;
  }

  /* move playershot-instance and check for collisions with gobj->rect */
  erg = VG3_move_object_check_collision(gmain, gmain->ofstruct, gmain->qdtr, objp, &gobj->rect, moving_factor, &gobj->xdelta, &gobj->ydelta, &gobj->xremainder, &gobj->yremainder);
  if (erg < 0) {  /* error: destroy playershot-instance */
    fprintf(stderr, "moving playershot: %s\n", VG3_error());
    f_free(gmain, objp);
    return;
  }
  if (erg == VGAG3_COLL_RETURN_DEAD) { return; }  /* instance has been freed */
  if (erg == VGAG3_COLL_RETURN_CATCHED) { f_free(gmain, objp); return; }

  /* insert playershot-instance into quadtree again */
  memset(&coll, 0, sizeof(coll));
  coll.rect = gobj->rect;
  snprintf(coll.oid, sizeof(coll.oid), "%s", objp->oid);
  coll.optr = objp;
  VG3_coll_q_insert(gmain->qdtr, &coll);
}


/* draw-function */
static void
f_draw(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_playershot *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_playershot *)objp->ostruct;
  if (gobj->strength == 0) { return; }

  /* draw playershot-instance */
  VG3_image_copy(gmain->wstruct, NULL, gobj->img, gobj->rect.x + gobj->rect.w / 2, gobj->rect.y + gobj->rect.h / 2, NULL, 0);
}
