#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-text.h"

void getofc_text(struct vg3_ofunc_objfunc *);

static struct vg3_ofunc_object * f_new(void *, unsigned int, ...);
static void f_free(void *, struct vg3_ofunc_object *);
static void f_draw(void *, struct vg3_ofunc_object *);
static int f_data(void *, struct vg3_ofunc_object *, void *);

#define MAX_SHOWCOUNT 5
#define MAX_LOOPCOUNT 30

/* +++ get-function +++ */

void
getofc_text(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_TEXT));
  ofc->f_new = f_new;
  ofc->f_free = f_free;
  ofc->f_draw = f_draw;
  ofc->f_data = f_data;
}


/* +++ object-instance-functions +++ */

/* new-function
 * variable parameters: none
 */
static struct vg3_ofunc_object *
f_new(void *vmain, unsigned int iparent, ...)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_text *gobj;
  va_list ap;

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* get arguments */
  va_start(ap, iparent);
  va_end(ap);

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }

  /* create and fill text-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_TEXT));
  objp->subid = 0;
  objp->drawlevel = 6;
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* insert text-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  return objp;
}


/* free-function */
static void
f_free(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_text *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_text *)objp->ostruct;

  /* remove text-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  /* free text-instance */
  free(gobj);
  free(objp);
}


/* draw-function */
static void
f_draw(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_text *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_text *)objp->ostruct;

  if (*gobj->text == '\0') { ; return; }

  gobj->loop_counter++;
  if (gobj->loop_counter >= gobj->max_loopcount) {
    if (++gobj->show_counter >= gobj->max_showcount) { *gobj->text = '\0'; }
    gobj->loop_counter = 0;
    return;
  }

  /* draw text-instance */
  if (gobj->loop_counter % gobj->max_loopcount < gobj->max_loopcount * 2 / 3) {
    VG3_text_simpledraw(gmain->wstruct, NULL, NULL, gmain->winw / 2, gmain->winh / 2, gobj->text, VGAG3_COLOR_YELLOW, VGAG3_COLOR_TRANSPARENT, 1);
  }
}


/* data-exchanging-function */
static int
f_data(void *vmain, struct vg3_ofunc_object *objp, void *vdata)
{
  struct g_main *gmain = vmain;
  struct fdata_number *fdnumber = (struct fdata_number *)vdata;
  struct g_obj_text *gobj;

  if (gmain == NULL || objp == NULL || fdnumber == NULL) { return 0; }

  gobj = (struct g_obj_text *)objp->ostruct;

  if (fdnumber->flag == FDATA_TEXT) {  /* show text */
    if (*gobj->text == '\0' || fdnumber->number3) {
      if (fdnumber->text != NULL && *fdnumber->text != '\0') {
        if (fdnumber->number1 > 0) {
          gobj->max_showcount = fdnumber->number1;
        } else {
          gobj->max_showcount = MAX_SHOWCOUNT;
        }
        if (fdnumber->number2 > 0) {
          gobj->max_loopcount = fdnumber->number2;
        } else {
          gobj->max_loopcount = MAX_LOOPCOUNT;
        }
        snprintf(gobj->text, sizeof(gobj->text), "%s", fdnumber->text);
        gobj->show_counter = gobj->loop_counter = 0;
      } else {
        *gobj->text = '\0';
      }
    }
  }

  return 1;
}
