#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <math.h>
#include <vgagames3.h>
#include "main.h"

#ifndef M_PI
# define M_PI 3.14159265358979323846
#endif

void getofmgmt_asteroid(struct vg3_ofunc_ofmgmt *);

/* static structure for this object-managing */
static struct {
  int is_activated;            /* whether activated */
  unsigned int *seed;          /* pointer to seed for random numbers */
  struct random_check *rdchk;  /* random-check */
  int freq;                    /* frequency of occurance of asteroid in permille */
  int pause;                   /* actual pausing value: 0 = no, >0 = decrementing each loop, <0 = waiting for activation */
} mgmt_asteroid;

static int f_activate(void *, struct vg3_ofunc *, va_list);
static void f_deactivate(void *, struct vg3_ofunc *);
static void f_run(void *, struct vg3_ofunc *);
static int f_data(void *, struct vg3_ofunc *, void *);


/* +++ get-function +++ */

/* fill out the passed structure for object-management-functions of this VgaGames3-object */
void
getofmgmt_asteroid(struct vg3_ofunc_ofmgmt *ofm) {
  if (ofm == NULL) { return; }
  snprintf(ofm->oid, sizeof(ofm->oid), "%s", get_oid_name(OID_NAME_ASTEROID));
  ofm->f_activate = f_activate;
  ofm->f_deactivate = f_deactivate;
  ofm->f_run = f_run;
  ofm->f_data = f_data;
}


/* +++ object-management-functions +++ */

/* activate the managing of asteroid-object
 * parameters in ap:
 * - int: frequency in permille
 * - int: pausing-value
 */
static int
f_activate(void *vmain, struct vg3_ofunc *ofstruct, va_list ap)
{
  struct g_main *gmain = vmain;

  if (gmain == NULL || ofstruct == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return -1; }
  (void)gmain;

  memset(&mgmt_asteroid, 0, sizeof(mgmt_asteroid));

  mgmt_asteroid.seed = VG3_ofunc_mgmt_get_seed(ofstruct, get_oid_name(OID_NAME_ASTEROID));

  mgmt_asteroid.rdchk = random_check_new(mgmt_asteroid.seed, 1);
  if (mgmt_asteroid.rdchk == NULL) { return -1; }

  /* get variadic parameters */
  mgmt_asteroid.freq = va_arg(ap, int);
  mgmt_asteroid.pause = va_arg(ap, int);

  if (mgmt_asteroid.freq < 0) { mgmt_asteroid.freq = 0; } else if (mgmt_asteroid.freq > 1000) { mgmt_asteroid.freq = 1000; }

  mgmt_asteroid.is_activated = 1;

  return 0;
}


/* deactivate the managing of asteroid-object */
static void
f_deactivate(void *vmain, struct vg3_ofunc *ofstruct)
{
  struct g_main *gmain = vmain;

  if (gmain == NULL || ofstruct == NULL) { return; }
  if (!mgmt_asteroid.is_activated) { return; }

  /* destroy all asteroid-objects */
  VG3_ofunc_objlist_call_free(ofstruct, gmain, get_oid_name(OID_NAME_ASTEROID));

  /* clean up the static structure */
  random_check_free(mgmt_asteroid.rdchk);
  memset(&mgmt_asteroid, 0, sizeof(mgmt_asteroid));
  mgmt_asteroid.is_activated = 0;
}


/* execute managing of asteroid-instances,
 * creating a new one, if not enough asteroid are active
 */
static void
f_run(void *vmain, struct vg3_ofunc *ofstruct)
{
  struct g_main *gmain = vmain;

  if (gmain == NULL || ofstruct == NULL) { return; }
  if (!mgmt_asteroid.is_activated) { return; }

  if (mgmt_asteroid.pause != 0) {
    if (mgmt_asteroid.pause > 0) { mgmt_asteroid.pause--; }
    return;
  }

  /* create new asteroid-instance? */
  if (random_check_exec(mgmt_asteroid.rdchk, (mgmt_asteroid.freq + 1) / 2)) {
    const int xplus = 30;
    const int x_da = gmain->winw * 2 / 3;
    const int x_weg = gmain->winw / 3;
    int y_da, y_weg, xrad, angle, boost;
    double degrad;
    int xm, ym, xd, yd, subid, spriteid;
    const struct vg3_ofunc_objfunc *ofc;

    ofc = VG3_ofunc_get_objfunc(ofstruct, get_oid_name(OID_NAME_ASTEROID));
    if (ofc == NULL) { mgmt_asteroid.freq = 0; return; }

    /* which asteroid */
    subid = (int)VG3_nw_random_getnext(1, 2, mgmt_asteroid.seed);
    if (subid == 1) { xrad = 8; } else { xrad = 20; }
    spriteid = (int)VG3_nw_random_getnext(1, 4, mgmt_asteroid.seed);

    /* moving angle */
    angle = (int)VG3_nw_random_getnext(0, 45, mgmt_asteroid.seed);
    degrad = (double)angle / 360. * (M_PI + M_PI);

    /* x-start position */
    xm = gmain->winw + xplus;

    /* y-start position: random position between a max and min value */
    y_da = xrad - (int)((double)(x_da + xplus) * tan(degrad));
    y_weg = gmain->winh - xrad - (int)((double)(x_weg + xplus) * tan(degrad));
    ym = y_da + (int)VG3_nw_random_getnext(0, y_weg - y_da, mgmt_asteroid.seed);

    /* moving speed for x- and y-direction from angle with a random boost */
    VG3_get_xy_direction_from_angle(360 - 90 -angle, &xd, &yd);
    boost = (int)VG3_nw_random_getnext(15, 35, mgmt_asteroid.seed);
    xd *= boost; xd /= 10;
    yd *= boost; yd /= 10;

    /* mirror y-position and y-speed? */
    if (VG3_nw_random_getnext(1, 2, mgmt_asteroid.seed) == 2) { ym = gmain->winh - ym; yd = -yd; }

    if (ofc->f_new(gmain, 0, xm, ym, xd, yd, subid, spriteid) == NULL) {
      fprintf(stderr, "%s\n", VG3_error());
      mgmt_asteroid.freq = 0;
      return;
    }
    mgmt_asteroid.pause = (int)VG3_nw_random_getnext(3, 6, mgmt_asteroid.seed);
  }
}


/* set variables */
static int
f_data(void *vmain, struct vg3_ofunc *ofstruct, void *vdata)
{
  struct g_main *gmain = vmain;
  struct fdata_number *fdnumber = (struct fdata_number *)vdata;

  if (gmain == NULL || ofstruct == NULL || fdnumber == NULL) { return 0; }
  if (!mgmt_asteroid.is_activated) { return 0; }
  (void)gmain;

  if (fdnumber->flag == FDATA_PAUSE) {  /* set pause */
    mgmt_asteroid.pause = fdnumber->number1;
  }

  return 1;
}
