#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"

void getofmgmt_factory(struct vg3_ofunc_ofmgmt *);

/* static structure for this object-managing */
static struct {
  int is_activated;            /* whether activated */
  unsigned int *seed;          /* pointer to seed for random numbers */
  int obj_max, obj_hit;        /* how many instances were created and were destroyed */
  int max_no;                  /* how many instances are to be created */
  int pause;                   /* actual pausing value: 0 = no, >0 = decrementing each loop, <0 = waiting for activation */
} mgmt_factory;

static int f_activate(void *, struct vg3_ofunc *, va_list);
static void f_deactivate(void *, struct vg3_ofunc *);
static void f_run(void *, struct vg3_ofunc *);
static int f_data(void *, struct vg3_ofunc *, void *);


/* +++ get-function +++ */

/* fill out the passed structure for object-management-functions of this VgaGames3-object */
void
getofmgmt_factory(struct vg3_ofunc_ofmgmt *ofm) {
  if (ofm == NULL) { return; }
  snprintf(ofm->oid, sizeof(ofm->oid), "%s", get_oid_name(OID_NAME_FACTORY));
  ofm->f_activate = f_activate;
  ofm->f_deactivate = f_deactivate;
  ofm->f_run = f_run;
  ofm->f_data = f_data;
}


/* +++ object-management-functions +++ */

/* activate the managing of factory-object
 * parameters in ap:
 * - int: how many instances are to be created
 */
static int
f_activate(void *vmain, struct vg3_ofunc *ofstruct, va_list ap)
{
  struct g_main *gmain = vmain;

  if (gmain == NULL || ofstruct == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return -1; }
  (void)gmain;

  memset(&mgmt_factory, 0, sizeof(mgmt_factory));

  mgmt_factory.seed = VG3_ofunc_mgmt_get_seed(ofstruct, get_oid_name(OID_NAME_FACTORY));

  /* get variadic parameters */
  mgmt_factory.max_no = va_arg(ap, int);

  if (mgmt_factory.max_no <= 0) { mgmt_factory.max_no = 1; }

  mgmt_factory.pause = 100;

  mgmt_factory.is_activated = 1;

  return 0;
}


/* deactivate the managing of factory-object */
static void
f_deactivate(void *vmain, struct vg3_ofunc *ofstruct)
{
  struct g_main *gmain = vmain;

  if (gmain == NULL || ofstruct == NULL) { return; }
  if (!mgmt_factory.is_activated) { return; }

  /* destroy all factory-objects */
  VG3_ofunc_objlist_call_free(ofstruct, gmain, get_oid_name(OID_NAME_FACTORY));

  /* clean up the static structure */
  memset(&mgmt_factory, 0, sizeof(mgmt_factory));
  mgmt_factory.is_activated = 0;
}


/* execute managing of factory-instances,
 * creating a new one, if not enough factories are active
 */
static void
f_run(void *vmain, struct vg3_ofunc *ofstruct)
{
  struct g_main *gmain = vmain;

  if (gmain == NULL || ofstruct == NULL) { return; }
  if (!mgmt_factory.is_activated) { return; }

  if (mgmt_factory.pause != 0) {
    if (mgmt_factory.pause > 0) { mgmt_factory.pause--; }
    return;
  }

  /* create new factory-instance? */
  if (mgmt_factory.obj_max < mgmt_factory.max_no) {
    int xm, ym, flag;
    const struct vg3_ofunc_objfunc *ofc;

    ofc = VG3_ofunc_get_objfunc(ofstruct, get_oid_name(OID_NAME_FACTORY));
    if (ofc == NULL) { mgmt_factory.obj_max = mgmt_factory.max_no; return; }

    xm = gmain->winw + 50;
    ym = (int)VG3_nw_random_getnext(50, gmain->winh - 50, mgmt_factory.seed);
    flag = (int)VG3_nw_random_getnext(1, 3, mgmt_factory.seed);
    if (ofc->f_new(gmain, 0, xm, ym, flag) == NULL) {
      fprintf(stderr, "%s\n", VG3_error());
      mgmt_factory.obj_max = mgmt_factory.max_no;
      return;
    }
    mgmt_factory.obj_max++;
    mgmt_factory.pause = (int)VG3_nw_random_getnext(120, 200, mgmt_factory.seed);
  }
}


/* set variables */
static int
f_data(void *vmain, struct vg3_ofunc *ofstruct, void *vdata)
{
  struct g_main *gmain = vmain;
  struct fdata_number *fdnumber = (struct fdata_number *)vdata;

  if (gmain == NULL || ofstruct == NULL || fdnumber == NULL) { return 0; }
  if (!mgmt_factory.is_activated) { return 0; }
  (void)gmain;

  if (fdnumber->flag == FDATA_PAUSE) {  /* set pause */
    mgmt_factory.pause = fdnumber->number1;
  } else if (fdnumber->flag == FDATA_HIT) {  /* set obj_hit */
    mgmt_factory.obj_hit++;
  } else if (fdnumber->flag == FDATA_RATIO) {  /* return obj_max and obj_hit */
    fdnumber->number1 = mgmt_factory.obj_max;
    fdnumber->number2 = mgmt_factory.obj_hit;
  } else if (fdnumber->flag == FDATA_DONE) {  /* whether all instances are through */
    if (mgmt_factory.obj_max < mgmt_factory.max_no) {
      fdnumber->number1 = 0;  /* no */
    } else {
      fdnumber->number1 = 1;  /* yes */
    }
    fdnumber->number2 = mgmt_factory.max_no;
    fdnumber->number3 = mgmt_factory.obj_max;
  }

  return 1;
}
