#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"

void getofmgmt_fighter(struct vg3_ofunc_ofmgmt *);

/* static structure for this object-managing */
static struct {
  int is_activated;            /* whether activated */
  unsigned int *seed;          /* pointer to seed for random numbers */
  struct {                     /* how many instances were created and were destroyed */
    int green_max, green_hit;    /* FIGHTER_GREEN */
    int yellow_max, yellow_hit;  /* FIGHTER_YELLOW */
    int red_max, red_hit;        /* FIGHTER_RED */
  } ratio;
  struct random_check *rdchk;  /* random-check */
  int stealth;                 /* whether stealth-fighter */
  int f_green;                 /* percent of green resp. (no)-stealth fighters */
  int f_yellow;                /* percent of yellow fighters */
  int f_red;                   /* percent of red fighters */
  int freq;                    /* frequency of occurance of fighters in permille */
  int pause;                   /* pausing: 0 = no, >0 = decrementing each loop, <0 = waiting for activation */
  int pausingbasis;            /* pausing basis value between fighters */
} mgmt_fighter;

static int f_activate(void *, struct vg3_ofunc *, va_list);
static void f_deactivate(void *, struct vg3_ofunc *);
static void f_run(void *, struct vg3_ofunc *);
static int f_data(void *, struct vg3_ofunc *, void *);


/* +++ get-function +++ */

/* fill out the passed structure for object-management-functions of this VgaGames3-object */
void
getofmgmt_fighter(struct vg3_ofunc_ofmgmt *ofm) {
  if (ofm == NULL) { return; }
  snprintf(ofm->oid, sizeof(ofm->oid), "%s", get_oid_name(OID_NAME_FIGHTER));
  ofm->f_activate = f_activate;
  ofm->f_deactivate = f_deactivate;
  ofm->f_run = f_run;
  ofm->f_data = f_data;
}


/* +++ object-management-functions +++ */

/* activate the managing of fighter-object
 * parameters in ap:
 * - int: stealth-value
 * - int: percent of green fighters
 * - int: percent of yellow fighters
 * - int: percent of red fighters
 * - int: frequency in permille
 * - int: pausing-value at start
 * - int: pausing-value-basis between fighters
 */
static int
f_activate(void *vmain, struct vg3_ofunc *ofstruct, va_list ap)
{
  struct g_main *gmain = vmain;
  int izahl;

  if (gmain == NULL || ofstruct == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return -1; }
  (void)gmain;

  memset(&mgmt_fighter, 0, sizeof(mgmt_fighter));

  mgmt_fighter.seed = VG3_ofunc_mgmt_get_seed(ofstruct, get_oid_name(OID_NAME_FIGHTER));

  mgmt_fighter.rdchk = random_check_new(mgmt_fighter.seed, 1);
  if (mgmt_fighter.rdchk == NULL) { return -1; }

  /* get variadic parameters */
  mgmt_fighter.stealth = va_arg(ap, int);
  mgmt_fighter.f_green = va_arg(ap, int);
  mgmt_fighter.f_yellow = va_arg(ap, int);
  mgmt_fighter.f_red = va_arg(ap, int);
  mgmt_fighter.freq = va_arg(ap, int);
  mgmt_fighter.pause = va_arg(ap, int);
  mgmt_fighter.pausingbasis = va_arg(ap, int);

  if (mgmt_fighter.stealth < 0 || mgmt_fighter.stealth > 2) { mgmt_fighter.stealth = 0; }
  if (mgmt_fighter.f_green < 0) { mgmt_fighter.f_green = 0; } else if (mgmt_fighter.f_green > 100) { mgmt_fighter.f_green = 100; }
  if (mgmt_fighter.f_yellow < 0) { mgmt_fighter.f_yellow = 0; } else if (mgmt_fighter.f_yellow > 100) { mgmt_fighter.f_yellow = 100; }
  if (mgmt_fighter.f_red < 0) { mgmt_fighter.f_red = 0; } else if (mgmt_fighter.f_red > 100) { mgmt_fighter.f_red = 100; }
  if (mgmt_fighter.freq < 0) { mgmt_fighter.freq = 0; } else if (mgmt_fighter.freq > 1000) { mgmt_fighter.freq = 1000; }

  izahl = mgmt_fighter.f_green + mgmt_fighter.f_yellow + mgmt_fighter.f_red;
  if (izahl > 0) {
    mgmt_fighter.f_green = mgmt_fighter.f_green * 100 / izahl;
    mgmt_fighter.f_yellow = mgmt_fighter.f_yellow * 100 / izahl;
    mgmt_fighter.f_red = mgmt_fighter.f_red * 100 / izahl;
  }

  mgmt_fighter.is_activated = 1;

  return 0;
}


/* deactivate the managing of fighter-object */
static void
f_deactivate(void *vmain, struct vg3_ofunc *ofstruct)
{
  struct g_main *gmain = vmain;

  if (gmain == NULL || ofstruct == NULL) { return; }
  if (!mgmt_fighter.is_activated) { return; }

  /* destroy all fighter-objects */
  VG3_ofunc_objlist_call_free(ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER));

  /* clean up the static structure */
  random_check_free(mgmt_fighter.rdchk);
  memset(&mgmt_fighter, 0, sizeof(mgmt_fighter));
  mgmt_fighter.is_activated = 0;
}


/* execute managing of fighter-instances,
 * creating a new one, if not enough asteroids are active
 */
static void
f_run(void *vmain, struct vg3_ofunc *ofstruct)
{
  struct g_main *gmain = vmain;

  if (gmain == NULL || ofstruct == NULL) { return; }
  if (!mgmt_fighter.is_activated) { return; }

  if (mgmt_fighter.pause != 0) {
    if (mgmt_fighter.pause > 0) { mgmt_fighter.pause--; }
    return;
  }

  /* create new fighter-instance? */
  if (random_check_exec(mgmt_fighter.rdchk, (mgmt_fighter.freq + 1) / 2)) {
    int f_sel, prz;

    /* select fighter */
    prz = random_check_exec(mgmt_fighter.rdchk, (mgmt_fighter.f_green + mgmt_fighter.f_yellow + mgmt_fighter.f_red) * 10);
    if (prz <= mgmt_fighter.f_green * 10) {
      if (mgmt_fighter.stealth) { f_sel = FIGHTER_STEALTH; } else { f_sel = FIGHTER_GREEN; }
    } else if (prz <= (mgmt_fighter.f_green + mgmt_fighter.f_yellow) * 10) {
      f_sel = FIGHTER_YELLOW;
    } else if (prz <= (mgmt_fighter.f_green + mgmt_fighter.f_yellow + mgmt_fighter.f_red) * 10) {
      f_sel = FIGHTER_RED;
    } else {
      f_sel = 0;
    }

    if (f_sel > 0) {
      int xm, ym;
      const struct vg3_ofunc_objfunc *ofc = VG3_ofunc_get_objfunc(ofstruct, get_oid_name(OID_NAME_FIGHTER));
      if (ofc == NULL) { mgmt_fighter.freq = 0; return; }

      xm = gmain->winw + 30;
      ym = (int)VG3_nw_random_getnext(10, gmain->winh - 10, mgmt_fighter.seed);
      if (ofc->f_new(gmain, 0, xm, ym, f_sel) == NULL) {
        fprintf(stderr, "%s\n", VG3_error());
        mgmt_fighter.freq = 0;
        return;
      }
      if (f_sel == FIGHTER_GREEN || f_sel == FIGHTER_STEALTH) {
        mgmt_fighter.ratio.green_max++;
      } else if (f_sel == FIGHTER_YELLOW) {
        mgmt_fighter.ratio.yellow_max++;
      } else if (f_sel == FIGHTER_RED) {
        mgmt_fighter.ratio.red_max++;
      }
      if (mgmt_fighter.pausingbasis > 0) {
        mgmt_fighter.pause = mgmt_fighter.pausingbasis
                             + (int)VG3_nw_random_getnext(0, mgmt_fighter.pausingbasis / 3, mgmt_fighter.seed)
                             - (mgmt_fighter.pausingbasis / 3 / 2);
      }
    }
  }
}


/* set variables */
static int
f_data(void *vmain, struct vg3_ofunc *ofstruct, void *vdata)
{
  struct g_main *gmain = vmain;
  struct fdata_number *fdnumber = (struct fdata_number *)vdata;

  if (gmain == NULL || ofstruct == NULL || fdnumber == NULL) { return 0; }
  if (!mgmt_fighter.is_activated) { return 0; }
  (void)gmain;

  if (fdnumber->flag == FDATA_PAUSE) {  /* set pause */
    mgmt_fighter.pause = fdnumber->number1;

  } else if (fdnumber->flag == FDATA_FREQ) {  /* set freq and pausingbasis */
    if (fdnumber->number1 >= 0) {
      mgmt_fighter.freq = fdnumber->number1;
      if (mgmt_fighter.freq > 1000) { mgmt_fighter.freq = 1000; }
    }
    if (fdnumber->number2 >= 0) {
      mgmt_fighter.pausingbasis = fdnumber->number2;
    }
  } else if (fdnumber->flag == FDATA_HIT) {  /* set hit-ratio */
    if (fdnumber->number1 == FIGHTER_GREEN || fdnumber->number1 == FIGHTER_STEALTH) {
      mgmt_fighter.ratio.green_hit++;
    } else if (fdnumber->number1 == FIGHTER_YELLOW) {
      mgmt_fighter.ratio.yellow_hit++;
    } else if (fdnumber->number1 == FIGHTER_RED) {
      mgmt_fighter.ratio.red_hit++;
    }
  } else if (fdnumber->flag == FDATA_RATIO) {  /* return max- and hit-ratio */
    if (fdnumber->number1 == FIGHTER_GREEN || fdnumber->number1 == FIGHTER_STEALTH) {
      fdnumber->number1 = mgmt_fighter.ratio.green_max;
      fdnumber->number2 = mgmt_fighter.ratio.green_hit;
    } else if (fdnumber->number1 == FIGHTER_YELLOW) {
      fdnumber->number1 = mgmt_fighter.ratio.yellow_max;
      fdnumber->number2 = mgmt_fighter.ratio.yellow_hit;
    } else if (fdnumber->number1 == FIGHTER_RED) {
      fdnumber->number1 = mgmt_fighter.ratio.red_max;
      fdnumber->number2 = mgmt_fighter.ratio.red_hit;
    } else {
      fdnumber->number1 = fdnumber->number2 = 0;
    }
  } else if (fdnumber->flag == FDATA_REINIT) {  /* reinit */
    mgmt_fighter.stealth = fdnumber->number1;
    mgmt_fighter.f_green = fdnumber->number2;
    mgmt_fighter.f_yellow = fdnumber->number3;
    mgmt_fighter.f_red = fdnumber->number4;
    mgmt_fighter.freq = fdnumber->number5;
    mgmt_fighter.pause = fdnumber->number6;
    mgmt_fighter.pausingbasis = fdnumber->number7;
  }

  return 1;
}
