#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"

void getofmgmt_radar(struct vg3_ofunc_ofmgmt *);

/* static structure for this object-managing */
static struct {
  int is_activated;            /* whether activated */
  unsigned int *seed;          /* pointer to seed for random numbers */
  int obj_max, obj_hit;        /* how many instances were created and were destroyed */
  struct random_check *rdchk;  /* random-check */
  int freq;                    /* frequency of occurance of radar in permille */
  int pause;                   /* pausing: 0 = no, >0 = decrementing each loop, <0 = waiting for activation */
} mgmt_radar;

static int f_activate(void *, struct vg3_ofunc *, va_list);
static void f_deactivate(void *, struct vg3_ofunc *);
static void f_run(void *, struct vg3_ofunc *);
static int f_data(void *, struct vg3_ofunc *, void *);


/* +++ get-function +++ */

/* fill out the passed structure for object-management-functions of this VgaGames3-object */
void
getofmgmt_radar(struct vg3_ofunc_ofmgmt *ofm) {
  if (ofm == NULL) { return; }
  snprintf(ofm->oid, sizeof(ofm->oid), "%s", get_oid_name(OID_NAME_RADAR));
  ofm->f_activate = f_activate;
  ofm->f_deactivate = f_deactivate;
  ofm->f_run = f_run;
  ofm->f_data = f_data;
}


/* +++ object-management-functions +++ */

/* activate the managing of radar-object
 * parameters in ap:
 * - int: frequency in permille
 * - int: pausing-value
 */
static int
f_activate(void *vmain, struct vg3_ofunc *ofstruct, va_list ap)
{
  struct g_main *gmain = vmain;

  if (gmain == NULL || ofstruct == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return -1; }
  (void)gmain;

  memset(&mgmt_radar, 0, sizeof(mgmt_radar));

  mgmt_radar.seed = VG3_ofunc_mgmt_get_seed(ofstruct, get_oid_name(OID_NAME_RADAR));

  mgmt_radar.rdchk = random_check_new(mgmt_radar.seed, 1);
  if (mgmt_radar.rdchk == NULL) { return -1; }

  /* get variadic parameters */
  mgmt_radar.freq = va_arg(ap, int);
  mgmt_radar.pause = va_arg(ap, int);

  if (mgmt_radar.freq < 0) { mgmt_radar.freq = 0; } else if (mgmt_radar.freq > 1000) { mgmt_radar.freq = 1000; }

  mgmt_radar.is_activated = 1;

  return 0;
}


/* deactivate the managing of radar-object */
static void
f_deactivate(void *vmain, struct vg3_ofunc *ofstruct)
{
  struct g_main *gmain = vmain;

  if (gmain == NULL || ofstruct == NULL) { return; }
  if (!mgmt_radar.is_activated) { return; }

  /* destroy all radar-objects */
  VG3_ofunc_objlist_call_free(ofstruct, gmain, get_oid_name(OID_NAME_RADAR));

  /* clean up the static structure */
  random_check_free(mgmt_radar.rdchk);
  memset(&mgmt_radar, 0, sizeof(mgmt_radar));
  mgmt_radar.is_activated = 0;
}


/* execute managing of radar-instances,
 * creating a new one, if not enough radars are active
 */
static void
f_run(void *vmain, struct vg3_ofunc *ofstruct)
{
  struct g_main *gmain = vmain;

  if (gmain == NULL || ofstruct == NULL) { return; }
  if (!mgmt_radar.is_activated) { return; }

  if (mgmt_radar.pause != 0) {
    if (mgmt_radar.pause > 0) { mgmt_radar.pause--; }
    return;
  }

  /* create new radar-instance? */
  if (random_check_exec(mgmt_radar.rdchk, (mgmt_radar.freq + 1) / 2)) {
    int xm, ym;
    const struct vg3_ofunc_objfunc *ofc;

    ofc = VG3_ofunc_get_objfunc(ofstruct, get_oid_name(OID_NAME_RADAR));
    if (ofc == NULL) { mgmt_radar.freq = 0; return; }

    xm = gmain->winw + 50;
    ym = (int)VG3_nw_random_getnext(10, gmain->winh - 10, mgmt_radar.seed);
    if (ofc->f_new(gmain, 0, xm, ym) == NULL) {
      fprintf(stderr, "%s\n", VG3_error());
      mgmt_radar.freq = 0;
      return;
    }
    mgmt_radar.obj_max++;
    mgmt_radar.pause = (int)VG3_nw_random_getnext(80, 250, mgmt_radar.seed);
  }
}


/* set variables */
static int
f_data(void *vmain, struct vg3_ofunc *ofstruct, void *vdata)
{
  struct g_main *gmain = vmain;
  struct fdata_number *fdnumber = (struct fdata_number *)vdata;

  if (gmain == NULL || ofstruct == NULL || fdnumber == NULL) { return 0; }
  if (!mgmt_radar.is_activated) { return 0; }
  (void)gmain;

  if (fdnumber->flag == FDATA_PAUSE) {  /* set pause */
    mgmt_radar.pause = fdnumber->number1;
  } else if (fdnumber->flag == FDATA_HIT) {  /* set obj_hit */
    mgmt_radar.obj_hit++;
  } else if (fdnumber->flag == FDATA_RATIO) {  /* return obj_max and obj_hit */
    fdnumber->number1 = mgmt_radar.obj_max;
    fdnumber->number2 = mgmt_radar.obj_hit;
  }

  return 1;
}
