#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"

void getofmgmt_tanker(struct vg3_ofunc_ofmgmt *);

/* static structure for this object-managing */
static struct {
  int is_activated;            /* whether activated */
  unsigned int *seed;          /* pointer to seed for random numbers */
  int obj_max, obj_hit;        /* how many instances were created and were destroyed */
  int max_no;                  /* how many instances are to be created */
  int pause;                   /* actual pausing value: 0 = no, >0 = decrementing each loop, <0 = waiting for activation */
} mgmt_tanker;

static int f_activate(void *, struct vg3_ofunc *, va_list);
static void f_deactivate(void *, struct vg3_ofunc *);
static void f_run(void *, struct vg3_ofunc *);
static int f_data(void *, struct vg3_ofunc *, void *);


/* +++ get-function +++ */

/* fill out the passed structure for object-management-functions of this VgaGames3-object */
void
getofmgmt_tanker(struct vg3_ofunc_ofmgmt *ofm) {
  if (ofm == NULL) { return; }
  snprintf(ofm->oid, sizeof(ofm->oid), "%s", get_oid_name(OID_NAME_TANKER));
  ofm->f_activate = f_activate;
  ofm->f_deactivate = f_deactivate;
  ofm->f_run = f_run;
  ofm->f_data = f_data;
}


/* +++ object-management-functions +++ */

/* activate the managing of tanker-object
 * parameters in ap:
 * - int: how many instances are to be created
 */
static int
f_activate(void *vmain, struct vg3_ofunc *ofstruct, va_list ap)
{
  struct g_main *gmain = vmain;

  if (gmain == NULL || ofstruct == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return -1; }
  (void)gmain;

  memset(&mgmt_tanker, 0, sizeof(mgmt_tanker));

  mgmt_tanker.seed = VG3_ofunc_mgmt_get_seed(ofstruct, get_oid_name(OID_NAME_TANKER));

  /* get variadic parameters */
  mgmt_tanker.max_no = va_arg(ap, int);

  if (mgmt_tanker.max_no <= 0) { mgmt_tanker.max_no = 1; }

  mgmt_tanker.pause = 250;

  mgmt_tanker.is_activated = 1;

  return 0;
}


/* deactivate the managing of tanker-object */
static void
f_deactivate(void *vmain, struct vg3_ofunc *ofstruct)
{
  struct g_main *gmain = vmain;

  if (gmain == NULL || ofstruct == NULL) { return; }
  if (!mgmt_tanker.is_activated) { return; }

  /* destroy all tanker-objects */
  VG3_ofunc_objlist_call_free(ofstruct, gmain, get_oid_name(OID_NAME_TANKER));

  /* clean up the static structure */
  memset(&mgmt_tanker, 0, sizeof(mgmt_tanker));
  mgmt_tanker.is_activated = 0;
}


/* execute managing of tanker-instances,
 * creating a new one, if not enough tankers are active
 */
static void
f_run(void *vmain, struct vg3_ofunc *ofstruct)
{
  struct g_main *gmain = vmain;

  if (gmain == NULL || ofstruct == NULL) { return; }
  if (!mgmt_tanker.is_activated) { return; }

  if (mgmt_tanker.pause != 0) {
    if (mgmt_tanker.pause > 0) { mgmt_tanker.pause--; }
    return;
  }

  /* create new tanker-instance? */
  if (mgmt_tanker.obj_max < mgmt_tanker.max_no) {
    int xm, ym, xd, yd;
    const struct vg3_ofunc_objfunc *ofc;

    ofc = VG3_ofunc_get_objfunc(ofstruct, get_oid_name(OID_NAME_TANKER));
    if (ofc == NULL) { mgmt_tanker.obj_max = mgmt_tanker.max_no; return; }

    xm = gmain->winw + 80;
    ym = (int)VG3_nw_random_getnext(20, gmain->winh - 20, mgmt_tanker.seed);
    xd = -(int)VG3_nw_random_getnext(5, 10, mgmt_tanker.seed);
    yd = (int)VG3_nw_random_getnext(0, 6, mgmt_tanker.seed) - 3;
    if (ym < gmain->winh / 3) {
      if (yd < 0) { yd = -yd; }
    } else if (ym > gmain->winh * 2 / 3) {
      if (yd > 0) { yd = -yd; }
    }
    if (ofc->f_new(gmain, 0, xm, ym, xd, yd) == NULL) {
      fprintf(stderr, "%s\n", VG3_error());
      mgmt_tanker.obj_max = mgmt_tanker.max_no;
      return;
    }
    mgmt_tanker.obj_max++;
    mgmt_tanker.pause = (int)VG3_nw_random_getnext(260, 350, mgmt_tanker.seed);
  }
}


/* set variables */
static int
f_data(void *vmain, struct vg3_ofunc *ofstruct, void *vdata)
{
  struct g_main *gmain = vmain;
  struct fdata_number *fdnumber = (struct fdata_number *)vdata;

  if (gmain == NULL || ofstruct == NULL || fdnumber == NULL) { return 0; }
  if (!mgmt_tanker.is_activated) { return 0; }
  (void)gmain;

  if (fdnumber->flag == FDATA_PAUSE) {  /* set pause */
    mgmt_tanker.pause = fdnumber->number1;
  } else if (fdnumber->flag == FDATA_HIT) {  /* set obj_hit */
    mgmt_tanker.obj_hit++;
  } else if (fdnumber->flag == FDATA_RATIO) {  /* return obj_max and obj_hit */
    fdnumber->number1 = mgmt_tanker.obj_max;
    fdnumber->number2 = mgmt_tanker.obj_hit;
  } else if (fdnumber->flag == FDATA_DONE) {  /* whether all instances are through */
    if (mgmt_tanker.obj_max < mgmt_tanker.max_no) {
      fdnumber->number1 = 0;  /* no */
    } else {
      fdnumber->number1 = 1;  /* yes */
    }
    fdnumber->number2 = mgmt_tanker.max_no;
    fdnumber->number3 = mgmt_tanker.obj_max;
  }

  return 1;
}
