#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"

int topten_show(void *, int);
int topten_insert(void *, int);
void topten_save(void *, FILE *);
void topten_read(void *, FILE *);


/* topten_show:
 * show top ten
 * @param vmain     main game struct (struct g_main)
 * @param showno    highlight showno entry, or 0 = no highlight
 * @return          0 = OK, 1 = exit-request
 */
int
topten_show(void *vmain, int showno)
{
  struct g_main *gmain = (struct g_main *)vmain;
  int retw, fonthd, fwidth, fwpadd, fheight, fhpadd, ypos;
  struct vg3_multilang *mlang;
  char zeile[128];
  const char *kptr1, *kptr2;
  int maxzch, xstart, ystart, zplus, idx, waitsec;
  struct vg3_image *bgimage;

#ifndef GAMESKEL_TOPTEN
  return 0;
#endif
  if (gmain == NULL) { return 0; }

  fonthd = VG3_font_highdouble(gmain->wstruct, 1);

  VG3_fontsize(gmain->wstruct, NULL, &fwidth, &fheight, &fwpadd, &fhpadd);
  xstart = (fwidth + fwpadd) * 5;
  ystart = (fheight + fwpadd) * 3;

  maxzch = (gmain->winw - xstart * 2) / (fwidth + fwpadd);
  if (maxzch >= (int)sizeof(zeile)) { maxzch = (int)sizeof(zeile) - 1; }

  mlang = VG3_multilang_new(NULL, "en");
  VG3_multilang_add(mlang, FILES_DIR "/mlang/topten");
  
  bgimage = NULL;
  if (GAMESKEL_TOPTEN_BGFILE != NULL) {
    bgimage = VG3_image_load(gmain->wstruct, GAMESKEL_TOPTEN_BGFILE, 1);
    if (bgimage == NULL) { fprintf(stderr, "%s\n", VG3_error()); }
  }

  if (showno > 0) { waitsec = 50; } else { waitsec = 0; }

  retw = 0;
  VG3_discard_input(gmain->wstruct);
  for (;;) {
    if (VG3_inputevent_update(gmain->wstruct)) { retw = 1; break; }

    if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_SPACE, VGAG3_IS_NEW_PRESSED)) { break; }

    VG3_draw_clear(gmain->wstruct, NULL, VGAG3_COLOR_BLACK);

    if (bgimage != NULL) { VG3_image_copy(gmain->wstruct, NULL, bgimage, gmain->winw / 2, gmain->winh / 2, NULL, 0); }

    /* title */
    ypos = ystart;
    VG3_text_simpledraw(gmain->wstruct, NULL, NULL,
                        gmain->winw / 2, ypos, 
                        VG3_multilang_get(mlang, "topten-title"),
                        VGAG3_COLOR_YELLOW, VGAG3_COLOR_BLUE, 0);

    /* header */
    ypos += fheight + fhpadd + ystart / 2;
    kptr1 = VG3_multilang_get(mlang, "topten-hname"),
    kptr2 = VG3_multilang_get(mlang, "topten-hscore"),
    zplus = maxzch - strlen(kptr1) - strlen(kptr2);
    if (zplus <= 0) { zplus = 1; }
    snprintf(zeile, maxzch + 1, "%s%*s%s", kptr1, zplus, " ", kptr2);
    VG3_text_simpledraw(gmain->wstruct, NULL, NULL, gmain->winw / 2, ypos, zeile, VGAG3_COLOR_YELLOW, VGAG3_COLOR_TRANSPARENT, 0);

    /* underline header */
    ypos += fheight + fhpadd;
    memset(zeile, '-', maxzch); zeile[maxzch] = '\0';
    VG3_text_simpledraw(gmain->wstruct, NULL, NULL, gmain->winw / 2, ypos, zeile, VGAG3_COLOR_YELLOW, VGAG3_COLOR_TRANSPARENT, 0);

    /* entries */
    for (idx = 0; idx < TOPTEN_MAX; idx++) {
      ypos += fheight + fhpadd + 1 + VG3_font_highdouble(gmain->wstruct, -1);
      if (*gmain->topten.n[idx].name != '\0') {
        kptr1 = gmain->topten.n[idx].name;
        kptr2 = gmain->topten.n[idx].score;
      } else {
        kptr1 = kptr2 = "-";
      }
      zplus = maxzch - strlen(kptr1) - strlen(kptr2);
      if (zplus <= 0) { zplus = 1; }
      snprintf(zeile, maxzch + 1, "%s%*s%s", kptr1, zplus, " ", kptr2);
      if (showno == idx + 1) {
        VG3_text_simpledraw(gmain->wstruct, NULL, NULL, gmain->winw / 2, ypos, zeile, VGAG3_COLOR_BLACK, VGAG3_COLOR_YELLOW, 0);
      } else {
        VG3_text_simpledraw(gmain->wstruct, NULL, NULL, gmain->winw / 2, ypos, zeile, VGAG3_COLOR_YELLOW, VGAG3_COLOR_TRANSPARENT, 0);
      }
    }

    if (waitsec == 0) {
      /* press space */
      VG3_text_simpledraw(gmain->wstruct, NULL, NULL,
                          gmain->winw / 2, gmain->winh - fheight,
                          VG3_multilang_get(mlang, "topten-space"),
                          VGAG3_COLOR_VIOLET, VGAG3_COLOR_TRANSPARENT, 0);
    }

    VG3_window_update(gmain->wstruct, 0, 0);
    VG3_wait_time(100);
    if (waitsec > 0) {
      if (--waitsec == 0) { break; }
    }
  }
  VG3_discard_input(gmain->wstruct);

  if (bgimage != NULL) { VG3_image_unload(gmain->wstruct, bgimage); }
  VG3_font_highdouble(gmain->wstruct, fonthd);
  if (mlang != NULL) { VG3_multilang_free(mlang); }

  VG3_draw_clear(gmain->wstruct, NULL, VGAG3_COLOR_BLACK);
  VG3_window_update(gmain->wstruct, 0, 0);

  return retw;
}


/* topten_insert:
 * show top ten
 * @param vmain  main game struct (struct g_main)
 * @param score  score
 * @return       0 = OK, 1 = exit-request
 */
int
topten_insert(void *vmain, int score)
{
  struct g_main *gmain = (struct g_main *)vmain;
  int retw, fonthd, showidx;
  struct vg3_multilang *mlang;
  char name[32];
  const char *kptr;
  int idx, number;

#ifndef GAMESKEL_TOPTEN
  return 0;
#endif
  if (gmain == NULL) { return 0; }

  for (idx = 0; idx < TOPTEN_MAX; idx++) {
    if (*gmain->topten.n[idx].name == '\0') { break; }
    number = atoi(gmain->topten.n[idx].score);
    if (number < score) { break; }
  }
  if (idx == TOPTEN_MAX) { return 0; }
  showidx = idx + 1;

  fonthd = VG3_font_highdouble(gmain->wstruct, 1);

  mlang = VG3_multilang_new(NULL, "en");
  VG3_multilang_add(mlang, FILES_DIR "/mlang/topten");

  kptr = getenv("USER");
  if (kptr == NULL || *kptr == '\0') { kptr = getenv("LOGNAME"); }
  if (kptr != NULL) { snprintf(name, sizeof(name), "%s", kptr); }
  
  retw = VG3_input_box(gmain->wstruct,
                       VGAG3_COLOR_YELLOW, VGAG3_COLOR_BROWN,
                       VG3_multilang_get(mlang, "topten-insert"),
                       name, sizeof(name));
  if (retw == 1) {
    idx = showidx - 1;
    if (idx < TOPTEN_MAX - 1) {
      memmove(&gmain->topten.n[idx + 1], &gmain->topten.n[idx], sizeof(*gmain->topten.n) * (TOPTEN_MAX - idx - 1));
    }
    memset(&gmain->topten.n[idx], 0, sizeof(*gmain->topten.n));
    snprintf(gmain->topten.n[idx].name, sizeof(gmain->topten.n[0].name), "%s", name);
    snprintf(gmain->topten.n[idx].score, sizeof(gmain->topten.n[0].score), "%d", score);
    retw = 0;

  } else {
    if (retw == -1) { fprintf(stderr, "%s\n", VG3_error()); }
    if (retw == -2) { retw = 1; } else { retw = 0; }
  }

  VG3_font_highdouble(gmain->wstruct, fonthd);
  if (mlang != NULL) { VG3_multilang_free(mlang); }

  VG3_draw_clear(gmain->wstruct, NULL, VGAG3_COLOR_BLACK);
  VG3_window_update(gmain->wstruct, 0, 0);

  if (retw == 0) { topten_show(gmain, showidx); }

  return retw;
}


/* topten_save:
 * save top ten
 * @param vmain  main game struct (struct g_main)
 * @param ffp    opened save-file
 */
void
topten_save(void *vmain, FILE *ffp)
{
  struct g_main *gmain = (struct g_main *)vmain;
  int idx, max;
  size_t slen;

  if (gmain == NULL || ffp == NULL) { return; }

  for (idx = 0; idx < TOPTEN_MAX; idx++) {
    if (*gmain->topten.n[idx].name == '\0') { break; }
  }
  max = idx;
  fprintf(ffp, "%02d", max % 100);

  for (idx = 0; idx < max; idx++) {
    slen = strlen(gmain->topten.n[idx].name);
    fwrite(gmain->topten.n[idx].name, 1, slen + 1, ffp);
    slen = strlen(gmain->topten.n[idx].score);
    fwrite(gmain->topten.n[idx].score, 1, slen + 1, ffp);
  }

  fprintf(ffp, "\n");
}


/* topten_read:
 * read top ten
 * @param vmain  main game struct (struct g_main)
 * @param ffp    opened save-file
 */
void
topten_read(void *vmain, FILE *ffp)
{
  struct g_main *gmain = (struct g_main *)vmain;
  char buf[3], c;
  int max, idx, ipos, err;

  if (gmain == NULL || ffp == NULL) { return; }

  memset(&gmain->topten, 0, sizeof(gmain->topten));
  if (fread(buf, 1, 2, ffp) == 0) { return; }
  buf[2] = '\0';
  max = atoi(buf);

  err = 0;
  for (idx = 0; idx < max; idx++) {
    /* name */
    for (ipos = 0;; ipos++) {
      if (fread(&c, 1, 1, ffp) == 0) { err = 1; break; }
      gmain->topten.n[idx].name[ipos] = c;
      if (c == 0) { break; }
    }
    if (err) { break; }
    /* score */
    for (ipos = 0;; ipos++) {
      if (fread(&c, 1, 1, ffp) == 0) { err = 1; break; }
      gmain->topten.n[idx].score[ipos] = c;
      if (c == 0) { break; }
    }
    if (err) { break; }
  }
  if (!err && (fread(&c, 1, 1, ffp) == 0 || c != '\n')) { err = 1; }
  if (err) { memset(&gmain->topten, 0, sizeof(gmain->topten)); }
}
