#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-dice.h"

void getofc_dice(struct vg3_ofunc_objfunc *);

static struct vg3_ofunc_object * f_new(void *, unsigned int, ...);
static void f_free(void *, struct vg3_ofunc_object *);
static void f_run(void *, struct vg3_ofunc_object *);
static void f_draw(void *, struct vg3_ofunc_object *);


/* +++ get-function +++ */

void
getofc_dice(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_DICE));
  ofc->f_new = f_new;
  ofc->f_free = f_free;
  ofc->f_run = f_run;
  ofc->f_draw = f_draw;
}


/* +++ object-instance-functions +++ */

/* new-function
 * variable parameter: none
 */
static struct vg3_ofunc_object *
f_new(void *vmain, unsigned int iparent, ...)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_dice *gobj;
  va_list ap;

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* get arguments */
  va_start(ap, iparent);
  va_end(ap);

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  gobj->steps = 0;
  gobj->keysdiscarded = 0;

  /* create and fill dice-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_DICE));
  objp->drawlevel = 2;
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* insert dice-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  return objp;
}


/* free-function */
static void
f_free(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_dice *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_dice *)objp->ostruct;

  /* remove dice-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  /* free dice-instance */
  free(gobj);
  free(objp);
}


/* run-function */
static void
f_run(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_dice *gobj;
  struct vg3_actionstack_elem actionp;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_dice *)objp->ostruct;
  actionp = VG3_actionstack_get(gmain->astck);

  /* relevant? */
  if (actionp.stack_id == 0) { return; }
  if (actionp.name_id != ASTACK_DICE) { return; }

  if (!gobj->keysdiscarded) { VG3_discard_input(gmain->wstruct); gobj->keysdiscarded = 1; }

  /* increment dice-number */
  gobj->steps++;
  if (gobj->steps > 3) { gobj->steps = 1; }

  if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_SPACE, VGAG3_IS_NEW_PRESSED)) {
    /* push action ASTACK_SELECT for sub-ID ASTACK_SELECT_SUNNYBOY onto actionstack */
    struct vg3_hash *hparm = VG3_hash_new();
    VG3_hash_setint(hparm, "steps", sizeof("steps"), gobj->steps);
    VG3_hash_setint(hparm, "select", sizeof("select"), ASTACK_SELECT_SUNNYBOY);
    VG3_actionstack_push(gmain->astck, ASTACK_SELECT, hparm, 0);
    VG3_hash_free(hparm);
    /* reset */
    gobj->steps = 0;
    gobj->keysdiscarded = 0;
  }
}


/* draw-function */
static void
f_draw(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_dice *gobj;
  struct vg3_actionstack_elem actionp;
  char msg[256];

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_dice *)objp->ostruct;
  actionp = VG3_actionstack_get(gmain->astck);

  /* relevant? */
  if (actionp.stack_id == 0) { return; }
  if (actionp.name_id != ASTACK_DICE) { return; }

  /* draw dice-instance */
  snprintf(msg, sizeof(msg), "Calculating steps.\nStop it with space-key\n%d", gobj->steps);
  VG3_text_simpledraw(gmain->wstruct, NULL, NULL, gmain->winw / 2, gmain->winh / 2, msg, VGAG3_COLOR_WHITE, VGAG3_COLOR_TRANSPARENT, 1);
}
