#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-select.h"

void getofc_select(struct vg3_ofunc_objfunc *);

static struct vg3_ofunc_object * f_new(void *, unsigned int, ...);
static void f_free(void *, struct vg3_ofunc_object *);
static void f_run(void *, struct vg3_ofunc_object *);
static void f_draw(void *, struct vg3_ofunc_object *);


/* +++ get-function +++ */

void
getofc_select(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_SELECT));
  ofc->f_new = f_new;
  ofc->f_free = f_free;
  ofc->f_run = f_run;
  ofc->f_draw = f_draw;
}


/* +++ object-instance-functions +++ */

/* new-function
 * variable parameter:
 * - int: sub-ID
 */
static struct vg3_ofunc_object *
f_new(void *vmain, unsigned int iparent, ...)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_select *gobj;
  va_list ap;
  int subid;

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* get arguments */
  va_start(ap, iparent);
  subid = va_arg(ap, int);
  va_end(ap);

  if (subid < 0) { subid = 0; }

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  gobj->keysdiscarded = 0;

  /* create and fill select-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_SELECT));
  objp->subid = (unsigned short)subid;
  objp->drawlevel = 2;
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* insert select-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  return objp;
}


/* free-function */
static void
f_free(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_select *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_select *)objp->ostruct;

  /* remove select-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  /* free select-instance */
  free(gobj);
  free(objp);
}


/* run-function */
static void
f_run(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_select *gobj;
  struct vg3_actionstack_elem actionp;
  int keynumber;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_select *)objp->ostruct;
  actionp = VG3_actionstack_get(gmain->astck);

  /* relevant? */
  if (actionp.stack_id == 0) { return; }
  if (actionp.name_id != ASTACK_SELECT) { return; }
  if (VG3_hash_getint(actionp.hparm, "select", sizeof("select")) != objp->subid) { return; }

  if (!gobj->keysdiscarded) { VG3_discard_input(gmain->wstruct); gobj->keysdiscarded = 1; }

  /* check for pressed keys */
  keynumber = 0;
  if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_1, VGAG3_IS_NEW_PRESSED)) { keynumber = 1; }
  if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_2, VGAG3_IS_NEW_PRESSED)) { keynumber = 2; }

  /* act according to pressed key and sub-ID */
  if (keynumber > 0) {
    if (objp->subid == ASTACK_SELECT_SUNNYBOY) {  /* select sunnyboy */
      /* pop itself, then push action ASTACK_SUNNYBOY onto actionstack */
      struct vg3_hash *hparm = VG3_hash_new();
      int steps = VG3_hash_getint(actionp.hparm, "steps", sizeof("steps"));
      VG3_hash_setint(hparm, "steps", sizeof("steps"), steps);
      VG3_hash_setint(hparm, "number", sizeof("number"), keynumber);
      VG3_actionstack_pop(gmain->astck);
      VG3_actionstack_push(gmain->astck, ASTACK_SUNNYBOY, hparm, 0);
      VG3_hash_free(hparm);
      /* reset */
      gobj->keysdiscarded = 0;
    } else if (objp->subid == ASTACK_SELECT_DIRECTION) {  /* select moving-direction */
      /* set parameter "direction" of its parent-element, then pop itself */
      struct vg3_actionstack_elem actionpp = VG3_actionstack_parent(gmain->astck);
      if (actionpp.stack_id > 0 && actionpp.name_id == ASTACK_SUNNYBOY) {  /* correct parent */
        VG3_hash_setint(actionpp.hparm, "direction", sizeof("direction"), keynumber);
      }
      VG3_actionstack_pop(gmain->astck);
      /* reset */
      gobj->keysdiscarded = 0;
    }
  }
}


/* draw-function */
static void
f_draw(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_select *gobj;
  struct vg3_actionstack_elem actionp;
  char msg[256];

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_select *)objp->ostruct;
  (void)gobj;
  actionp = VG3_actionstack_get(gmain->astck);

  /* relevant? */
  if (actionp.stack_id == 0) { return; }
  if (actionp.name_id != ASTACK_SELECT) { return; }
  if (VG3_hash_getint(actionp.hparm, "select", sizeof("select")) != objp->subid) { return; }

  /* draw select-instance */
  if (objp->subid == ASTACK_SELECT_SUNNYBOY) {  /* select sunnyboy */
    snprintf(msg, sizeof(msg), "Select sunnyboy.\n[1] or [2]");
  } else if (objp->subid == ASTACK_SELECT_DIRECTION) {  /* select moving-direction */
    snprintf(msg, sizeof(msg), "Select direction.\n[1] or [2]");
  }
  VG3_text_simpledraw(gmain->wstruct, NULL, NULL, gmain->winw / 2, gmain->winh / 2, msg, VGAG3_COLOR_WHITE, VGAG3_COLOR_TRANSPARENT, 1);
}
