#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-sunnyboy.h"

void getofc_sunnyboy(struct vg3_ofunc_objfunc *);

static struct vg3_ofunc_object * f_new(void *, unsigned int, ...);
static void f_free(void *, struct vg3_ofunc_object *);
static void f_run(void *, struct vg3_ofunc_object *);
static void f_draw(void *, struct vg3_ofunc_object *);


/* +++ get-function +++ */

void
getofc_sunnyboy(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_SUNNYBOY));
  ofc->f_new = f_new;
  ofc->f_free = f_free;
  ofc->f_run = f_run;
  ofc->f_draw = f_draw;
}


/* +++ object-instance-functions +++ */

/* new-function
 * variable parameter:
 * - int: number of sunnyboy
 */
static struct vg3_ofunc_object *
f_new(void *vmain, unsigned int iparent, ...)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_sunnyboy *gobj;
  va_list ap;
  int number;
  char buf[128];

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* get arguments */
  va_start(ap, iparent);
  number = va_arg(ap, int);
  va_end(ap);
  if (number != 1 && number != 2) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  gobj->number = number;
  gobj->direction = 0;
  /* load image */
  snprintf(buf, sizeof(buf), "sunnyboy%d.bmp", gobj->number);
  gobj->img = VG3_image_load(gmain->wstruct, buf, 1);
  if (gobj->img == NULL) { return NULL; }
  /* set position-rectangle */
  VG3_image_getsize(gmain->wstruct, gobj->img, NULL, &gobj->rect.w, &gobj->rect.h);
  gobj->rect.x = gobj->rect.w / 2;
  if (gobj->number == 1) {
    gobj->rect.y = gmain->winh / 2 - gobj->rect.h / 2 - gobj->rect.h * 2;
  } else {
    gobj->rect.y = gmain->winh / 2 - gobj->rect.h / 2 + gobj->rect.h * 2;
  }

  /* create and fill sunnyboy-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_SUNNYBOY));
  objp->drawlevel = 1;
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* insert sunnyboy-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  return objp;
}


/* free-function */
static void
f_free(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_sunnyboy *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_sunnyboy *)objp->ostruct;

  /* remove sunnyboy-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  /* free sunnyboy-instance */
  if (gobj->img != NULL) { VG3_image_unload(gmain->wstruct, gobj->img); }
  free(gobj);
  free(objp);
}


/* run-function */
static void
f_run(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_sunnyboy *gobj;
  struct vg3_actionstack_elem actionp;
  int direction, steps;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_sunnyboy *)objp->ostruct;
  actionp = VG3_actionstack_get(gmain->astck);

  /* relevant? */
  if (actionp.stack_id == 0) { return; }
  if (actionp.name_id != ASTACK_SUNNYBOY) { return; }
  /* check for the correct sunnyboy-instance */
  if (VG3_hash_getint(actionp.hparm, "number", sizeof("number")) != gobj->number) { return; }

  /* update gobj->direction to "direction" in actionp.hparm if set */
  direction = VG3_hash_getint(actionp.hparm, "direction", sizeof("direction"));
  if (direction > 0) { gobj->direction = direction; }

  /* move sunnyboy */
  steps = VG3_hash_getint(actionp.hparm, "steps", sizeof("steps"));

  if (gobj->direction == 0) {  /* move to the right */
    if (gobj->rect.x > gmain->winw / 2) {  /* middle of window reached */
      /* push action ASTACK_SELECT for sub-ID ASTACK_SELECT_DIRECTION onto actionstack,
       * as this is an interruption of moving, to set the parameter "direction",
       * push it with the actual actionstack-element as parent 
       */
      struct vg3_hash *hparm = VG3_hash_new();
      VG3_hash_setint(hparm, "select", sizeof("select"), ASTACK_SELECT_DIRECTION);
      VG3_actionstack_push(gmain->astck, ASTACK_SELECT, hparm, actionp.stack_id);
      VG3_hash_free(hparm);
      /* nothing else to be done, as waiting for ASTACK_SELECT to fill parameter "direction" */
      return;
    }
    /* move */
    gobj->rect.x += gobj->rect.w;
  } else if (gobj->direction == 1) {  /* move upwards */
    gobj->rect.y -= gobj->rect.h;
  } else if (gobj->direction == 2) {  /* move downwards */
    gobj->rect.y += gobj->rect.h;
  }

  /* exit, if moved out of the window */
  if (gobj->rect.y < -gobj->rect.h || gobj->rect.y > gmain->winh + gobj->rect.h) {
    VG3_actionstack_pop(gmain->astck);  /* remove ASTACK_SUNNYBOY */
    gmain->exit = 1;
    return;
  }

  /* update parameter "steps" */
  steps--;
  VG3_hash_setint(actionp.hparm, "steps", sizeof("steps"), steps);
  if (steps == 0) {  /* moving done */
    VG3_actionstack_pop(gmain->astck);  /* remove ASTACK_SUNNYBOY */
  }
}


/* draw-function */
static void
f_draw(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_sunnyboy *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_sunnyboy *)objp->ostruct;

  /* always relevant */

  /* draw sunnyboy-instance */
  VG3_image_copy(gmain->wstruct, NULL, gobj->img, gobj->rect.x + gobj->rect.w / 2, gobj->rect.y + gobj->rect.h / 2, NULL, 0);
}
