/* Example for audio functions */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <vgagames3.h>

int main(int argc, char **argv) {
  /* Play an audio file "music.wav" until it has been ended
   * and check for key-strokes for pausing, changing volume and (un)muting
   */

  struct vg3_window *wstruct;
  int winw, winh;

  /* open window */
  wstruct = VG3_window_new(argv[0], VGAG3_VGAVERSION_LOW, VGAG3_WINSCALE_BESTSCALE);
  if (wstruct == NULL) { fprintf(stderr, "%s\n", VG3_error()); exit(1); }

  /* get the size of the window */
  VG3_window_getsize(wstruct, &winw, &winh);

  /* play a sound "music.wav" until it is over */
  { struct vg3_image *imgptr;
    struct vg3_text stxt;
    int audkz;
    int obpause = 0;
    const char *infotext =
      "[P] = pause/continue\n[Q] = quit\n[1] = -volume\n[2] = +volume\n[M] = (un)mute\n\nPlaying once...";

    /* create text-info image */
    VGAG3_TEXT_ATTRIBUTES_SET(&stxt, NULL, '\n', 0, infotext);
    imgptr = VG3_image_text(wstruct, &stxt, VGAG3_COLOR_YELLOW, VGAG3_COLOR_TRANSPARENT);
    if (imgptr == NULL) { fprintf(stderr, "%s\n", VG3_error()); goto byebye; }

    /* load "music.wav" and play it once with crescendo */
    audkz = VG3_audio_load(wstruct, "music.wav", 100, VGAG3_AUDIO_VOLUME_SOUND);
    if (audkz == 0) { fprintf(stderr, "loading music.wav: %s\n", VG3_error()); goto byebye; }
    VG3_audio_play(wstruct, audkz, 0, 1);

    /* copy the text-info image centered to the window */
    VG3_image_copy(wstruct, NULL, imgptr, winw / 2, winh / 2, NULL, 0);

    /* update window contents and wait for end of sound or pressing key */
    VG3_discard_input(wstruct);
    for (;;) {
      if (VG3_inputevent_update(wstruct) > 0) { break; }

      /* pause or continue playing sound */
      if (VG3_key_ispressed(wstruct, VGAG3_KEY_P, VGAG3_IS_NEW_PRESSED)) {
        obpause = !obpause;
        VG3_audio_pause(wstruct, audkz, obpause);
      }

      /* quit */
      if (VG3_key_ispressed(wstruct, VGAG3_KEY_Q, VGAG3_IS_NEW_PRESSED)) { break; }

      /* decrease volume */
      if (VG3_key_ispressed(wstruct, VGAG3_KEY_1, VGAG3_IS_PRESSED)) {
        int val = VG3_audio_volume(wstruct, VGAG3_AUDIO_VOLUME_SOUND, -1);
        VG3_audio_volume(wstruct, VGAG3_AUDIO_VOLUME_SOUND, val - 10);
      }

      /* increase volume */
      if (VG3_key_ispressed(wstruct, VGAG3_KEY_2, VGAG3_IS_PRESSED)) {
        int val = VG3_audio_volume(wstruct, VGAG3_AUDIO_VOLUME_SOUND, -1);
        VG3_audio_volume(wstruct, VGAG3_AUDIO_VOLUME_SOUND, val + 10);
      }

      /* (un)mute playing sound */
      if (VG3_key_ispressed(wstruct, VGAG3_KEY_M, VGAG3_IS_NEW_PRESSED)) {
        int val = VG3_audio_mute(wstruct, -1);
        VG3_audio_mute(wstruct, !val);
      }

      /* quit, if playing sound is ended */
      if (!VG3_audio_isplaying(wstruct, audkz)) { break; }

      VG3_window_update(wstruct, 0, 0);

      VG3_wait_time(50);
    }
    VG3_discard_input(wstruct);

    /* (stop and) unload the sound */
    VG3_audio_unload(wstruct, audkz);

    /* free the image */
    VG3_image_unload(wstruct, imgptr);
  }

  /* if an error occurred after creating the window, close it before exiting */
  byebye:
  /* close window */
  VG3_window_free(wstruct);

  exit(0);
}
