/* Example for sysmenu functions */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <vgagames3.h>

int main(int argc, char **argv) {
  /* create a system-menu with the simple system-menu functions;
   * for an example with individual system-menu entries refer
   * to VG3_sysmenu_insert_dataentry()
   */

  struct vg3_window *wstruct;

  /* open window */
  wstruct = VG3_window_new(argv[0], VGAG3_VGAVERSION_LOW, VGAG3_WINSCALE_NOSCALE);
  if (wstruct == NULL) { fprintf(stderr, "%s\n", VG3_error()); exit(1); }

  /* create system-menu and run game */
  { struct vg3_sysmenu *sysm;
    struct vg3_text stxt;
    void *inputmenu, *submenu;
    int *fjid, idx, default_key, keyidx;
    struct vg3_gamecontroller gcs;
    const char *keyname;
    enum { GOING_UP, GOING_DOWN, GOING_RIGHT, GOING_LEFT };

    /* create system-menu */

    sysm = VG3_sysmenu_new(wstruct, NULL, VG3_color_brightness(VGAG3_COLOR_GREEN, 50), VGAG3_COLOR_GREEN);

    /* insert exit-menu */
    VG3_sysmenu_simple_exitmenu(sysm, NULL, "Exit game");

    /* insert volume-menu with only main-, sound- and music-volume */
    VG3_sysmenu_simple_volumemenu(sysm, NULL, "Main volume", "Sound volume", "Music volume", NULL);

    /* insert window-menu for resizing the window */
    VG3_sysmenu_simple_windowmenu(sysm, NULL, "Change window");

    /* insert a sub-menu for keyboard- and gamecontroller-/joystick-settings */
    VGAG3_TEXT_ATTRIBUTES_SET(&stxt, NULL, 0, 0, "Input settings");
    inputmenu = VG3_sysmenu_insert_menuentry(sysm, NULL, &stxt);

    /* insert keyboard-menu into input settings sub-menu */
    submenu = VG3_sysmenu_simple_keyboardmenu(sysm, inputmenu);

    /* add cursor-key-settings to keyboard-menu */
    VG3_sysmenu_simple_keyboardentry(sysm, submenu, GOING_UP, "go up", VGAG3_KEY_UCURS, "Going up");
    VG3_sysmenu_simple_keyboardentry(sysm, submenu, GOING_DOWN, "go down", VGAG3_KEY_DCURS, "Going down");
    VG3_sysmenu_simple_keyboardentry(sysm, submenu, GOING_RIGHT, "go right", VGAG3_KEY_RCURS, "Going right");
    VG3_sysmenu_simple_keyboardentry(sysm, submenu, GOING_LEFT, "go left", VGAG3_KEY_LCURS, "Going left");

    /* insert gamecontroller-/joystick-menu into input settings sub-menu including sub-menus for each found device */
    submenu = VG3_sysmenu_simple_gcmenu(sysm, inputmenu);

    /* for each found gamecontroller/joystick add all four directions of the right axis */
    VG3_gamecontroller_getall(wstruct, &fjid);
    for (idx = 1; idx <= fjid[0]; idx++) {  /* for each device */
      /* right axis up */
      VG3_gamecontroller_getinfo(wstruct, fjid[idx], &gcs);
      if (gcs.is_gamecontroller) {
        default_key = VGAG3_GC_AXIS_RIGHTY_UP;
      } else {
        default_key = 1;  /* just a key */
      }
      VG3_sysmenu_simple_gcentry(sysm, submenu, fjid[idx], GOING_UP, "go up", default_key, "Going up");
      /* right axis down */
      VG3_gamecontroller_getinfo(wstruct, fjid[idx], &gcs);
      if (gcs.is_gamecontroller) {
        default_key = VGAG3_GC_AXIS_RIGHTY_DOWN;
      } else {
        default_key = 2;  /* just a key */
      }
      VG3_sysmenu_simple_gcentry(sysm, submenu, fjid[idx], GOING_DOWN, "go down", default_key, "Going down");
      /* right axis right */
      VG3_gamecontroller_getinfo(wstruct, fjid[idx], &gcs);
      if (gcs.is_gamecontroller) {
        default_key = VGAG3_GC_AXIS_RIGHTX_RIGHT;
      } else {
        default_key = 3;  /* just a key */
      }
      VG3_sysmenu_simple_gcentry(sysm, submenu, fjid[idx], GOING_RIGHT, "go right", default_key, "Going right");
      /* right axis left */
      VG3_gamecontroller_getinfo(wstruct, fjid[idx], &gcs);
      if (gcs.is_gamecontroller) {
        default_key = VGAG3_GC_AXIS_RIGHTX_LEFT;
      } else {
        default_key = 4;  /* just a key */
      }
      VG3_sysmenu_simple_gcentry(sysm, submenu, fjid[idx], GOING_LEFT, "go left", default_key, "Going left");
    }

    free(fjid);

    /* run game */

    printf("\nPress cursor keys or move right axis,\npress escape-key for system-menu,\npress return-key to quit\n");

    VG3_discard_input(wstruct);
    for (;;) {  /* game loop */
      if (VG3_inputevent_update(wstruct) > 0) { break; }

      /* check if return-key is pressed for exiting */
      if (VG3_key_ispressed(wstruct, VGAG3_KEY_ENTER, VGAG3_IS_NEW_PRESSED)) { break; }

      /* check for escape-key for entering system-menu */
      if (VG3_key_ispressed(wstruct, VGAG3_KEY_ESC, VGAG3_IS_NEW_PRESSED)) {
        if (VG3_sysmenu_exec(sysm)) { break; }
      }

      /* check for cursor-keys of keyboard */
      /* cursor-up */
      keyidx = VG3_sysmenu_getkeyval(sysm, 0, GOING_UP);
      keyname = VG3_key_num2name(wstruct, keyidx);
      if (VG3_key_ispressed(wstruct, keyidx, VGAG3_IS_PRESSED)) { printf("%s pressed\n", keyname); }
      /* cursor-down */
      keyidx = VG3_sysmenu_getkeyval(sysm, 0, GOING_DOWN);
      keyname = VG3_key_num2name(wstruct, keyidx);
      if (VG3_key_ispressed(wstruct, keyidx, VGAG3_IS_PRESSED)) { printf("%s pressed\n", keyname); }
      /* cursor-right */
      keyidx = VG3_sysmenu_getkeyval(sysm, 0, GOING_RIGHT);
      keyname = VG3_key_num2name(wstruct, keyidx);
      if (VG3_key_ispressed(wstruct, keyidx, VGAG3_IS_PRESSED)) { printf("%s pressed\n", keyname); }
      /* cursor-left */
      keyidx = VG3_sysmenu_getkeyval(sysm, 0, GOING_LEFT);
      keyname = VG3_key_num2name(wstruct, keyidx);
      if (VG3_key_ispressed(wstruct, keyidx, VGAG3_IS_PRESSED)) { printf("%s pressed\n", keyname); }

      /* check for right axis of each gamecontroller/joystick */
      VG3_gamecontroller_getall(wstruct, &fjid);
      for (idx = 1; idx <= fjid[0]; idx++) {  /* for each device */
        /* right-axis-up */
        keyidx = VG3_sysmenu_getkeyval(sysm, fjid[idx], GOING_UP);
        keyname = VG3_gamecontroller_num2name(wstruct, fjid[idx], keyidx);
        if (VG3_gamecontroller_ispressed(wstruct, fjid[idx], keyidx, VGAG3_IS_PRESSED)) {
          printf("%s pressed\n", keyname);
        }
        /* right-axis-down */
        keyidx = VG3_sysmenu_getkeyval(sysm, fjid[idx], GOING_DOWN);
        keyname = VG3_gamecontroller_num2name(wstruct, fjid[idx], keyidx);
        if (VG3_gamecontroller_ispressed(wstruct, fjid[idx], keyidx, VGAG3_IS_PRESSED)) {
          printf("%s pressed\n", keyname);
        }
        /* right-axis-right */
        keyidx = VG3_sysmenu_getkeyval(sysm, fjid[idx], GOING_RIGHT);
        keyname = VG3_gamecontroller_num2name(wstruct, fjid[idx], keyidx);
        if (VG3_gamecontroller_ispressed(wstruct, fjid[idx], keyidx, VGAG3_IS_PRESSED)) {
          printf("%s pressed\n", keyname);
        }
        /* right-axis-left */
        keyidx = VG3_sysmenu_getkeyval(sysm, fjid[idx], GOING_LEFT);
        keyname = VG3_gamecontroller_num2name(wstruct, fjid[idx], keyidx);
        if (VG3_gamecontroller_ispressed(wstruct, fjid[idx], keyidx, VGAG3_IS_PRESSED)) {
          printf("%s pressed\n", keyname);
        }
      }

      VG3_window_update(wstruct, 0, 0);

      /* wait up to 50 msec */
      VG3_wait_time(50);
    }

    VG3_discard_input(wstruct);
    VG3_sysmenu_free(sysm);
  }

  /* close window */
  VG3_window_free(wstruct);

  exit(0);
}
